/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#pragma once

#ifndef __PRIMITIVE_H__
#define __PRIMITIVE_H__

#include "BVHStatic.h"
#include "BVHMoving.h"
#include "BerconShadeContext.h"
#include "fieldFunctions.h"

#define BERCON_METABALL_PRIM_STATIC 0x10011256
#define BERCON_METABALL_PRIM_MOVING 0x10011257

#define NORMALDELTA 0.001f // Delta used in the normal computations below

inline void computeNormalLocations(VR::Vector* location, VR::Vector pos) {
	location[0] = pos;
	location[1] = pos - VR::Vector(NORMALDELTA,0.f,0.f);
	location[2] = pos - VR::Vector(0.f,NORMALDELTA,0.f);
	location[3] = pos - VR::Vector(0.f,0.f,NORMALDELTA);
}

inline void computeNormalResult(VR::Ireal* res) {
	res[1] = res[0] - res[1];
	res[2] = res[0] - res[2];
	res[3] = res[0] - res[3];
	VR::Ireal length = sqrt(res[1]*res[1] + res[2]*res[2] + res[3]*res[3]);
	res[1] /= length;
	res[2] /= length;
	res[3] /= length;
}

struct MyCacheStruct {	
	VR::Vector rO;
	VR::Vector col; // Surface color information
};

struct SurfaceInfo {
	VR::Vector col; // Surface color information
};

struct MetaballParams {
	float size;
	float threshold;
	float step;
	float error;
	int field;
	int color;
	bool useAverage;
	float cutoff;
	float trim;
	int depth;
	float leafLength;
	float leafSize;
	int steps;
	Texmap *tex;
	bool texType;
	float texStr;
	bool clamp;
	float clampTo;
	//Matrix3 tm;
	//Matrix3 itm;
};

//typedef VR::RayCache<MyCacheStruct> VR::RayCache<MyCacheStruct>;

/*inline float smoothstep(float d, float low, float high) { 
	d = (d-low)/(high-low);
	if (d < 0) return 0.f;
	if (d > 1) return 1.f;
	return (d*d*(3.f-2.f*d));
}*/

inline VR::Ireal smoothstep1(VR::Ireal d, VR::Ireal low) {
	d = (d - low) / (1.0 - low);
	if (d < 0) return 0.0;
	if (d > 1) return 1.0;
	return (d * d * (3.0 - 2.0 * d));
}

/*
	Common stuff between static and moving versions of the metaball primitive
	The primitive does the actual interscetion part
*/
class  BerconMetaballPrimitive {
public:
	// Parameters	
	VR::Ireal blobSize;	
	VR::Ireal blobThreshold;
	VR::Ireal stepLength;
	VR::Ireal maxError;
	VR::Ireal (*function)(VR::Ireal size2, VR::Ireal squaredDistance);
	VR::Ireal (*function2)(VR::Ireal size2, VR::Ireal squaredDistance);
	bool useAverage;	
	VR::Ireal cutoff;
	VR::Ireal trim;

	Texmap *tex;
	bool texType;
	VR::real texStr;
	VR::real texStr1; // 1.f - texStr

	bool clamp;
	VR::Ireal clampTo;
	//Matrix3 tm;
	//Matrix3 itm;

	VR::Vector *colors;

	// Generated values
	VR::Ireal blobSize2;

	// Cache
	VR::RayCache<MyCacheStruct> *raycache;
	
	// Methods
	inline void applyTexmap(VR::Vector pos, VR::Ireal& res) const {
		if (tex) {
			BerconShadeContext sc;
			sc.Init(Point3(pos.x, pos.y, pos.z));			
			float textureVal = tex->EvalMono(sc);
			if (clamp && res > clampTo) res = clampTo;
			if (texType) res -= textureVal * texStr; else res *= texStr1 + textureVal * texStr;
		}
	}
	
	inline void applyTexmap(BerconShadeContext& sc, VR::Vector pos, VR::Ireal& res) const {
		sc.Init(Point3(pos.x, pos.y, pos.z));			
		float textureVal = tex->EvalMono(sc);
		if (clamp && res > clampTo) res = clampTo;
		if (texType) res -= textureVal * texStr; else res *= texStr1 + textureVal * texStr;
	}
	
	inline void applyTexmap(VR::Ireal textureVal, VR::Ireal& res) const {
		if (clamp && res > clampTo) res = clampTo;
		if (texType) res -= textureVal; else res *= textureVal; // texStr has been precomputed in getTexmap()
	}
	
	inline VR::Ireal getTexmap(BerconShadeContext& sc, VR::Vector pos) {
		sc.Init(Point3(pos.x, pos.y, pos.z));			
		if (texType)
			return tex->EvalMono(sc) * texStr;
		return texStr1 + tex->EvalMono(sc) * texStr;
	}

	inline bool testThreshold(VR::Ireal f, bool insideField) { return insideField ? f < blobThreshold : f > blobThreshold; }

	void getSurfaceInfo(SurfaceInfo* s, VR::RSRay &ray) {
		MyCacheStruct cacheStruct;
		raycache->getCache(*(VR::VRayContext*) ray.rayparams, cacheStruct);	
		s->col = cacheStruct.col;
	}

	/*
	void getSurfaceInfo(SurfaceInfo* s, VR::RSRayRef& rayref) {
		MyCacheStruct cacheStruct;
		raycache->getCache(*(VR::VRayContext*) rayref.getRayParams(), cacheStruct);
		s->col = cacheStruct.col;
	}
	*/

	void setFunction(int x, int y) {
		switch (x) {
			case 0: this->function = &fieldFunctionPower2;	break;
			case 1: this->function = &fieldFunctionPower3;	break;
			case 2: this->function = &fieldFunctionPower4;	break;
			case 3: this->function = &fieldFunctionPower5;	break;
			case 4: this->function = &fieldFunctionHermit2;	break;
			case 5: this->function = &fieldFunctionHermit4;	break;		
			default: this->function = &fieldFunctionWyvill; break;
		}
		switch (y) {
			case 0: this->function2 = &fieldFunctionPower2;	break;
			case 1: this->function2 = &fieldFunctionPower3;	break;
			case 2: this->function2 = &fieldFunctionPower4;	break;
			case 3: this->function2 = &fieldFunctionPower5;	break;
			case 4: this->function2 = &fieldFunctionHermit2;	break;
			case 5: this->function2 = &fieldFunctionHermit4;	break;		
			default: this->function2 = &fieldFunctionWyvill; break;
		}	
	}

	void setParams(MetaballParams params) {
		this->blobSize = params.size;
		this->blobSize2 = params.size * params.size; // We'll be using this a whole lot, better cache it
		this->blobThreshold = params.threshold;
		this->stepLength = params.step;
		this->cutoff = params.cutoff;
		this->trim = params.trim;
		this->maxError = params.error;
		this->useAverage = params.useAverage;

		this->tex = params.tex;
		this->texType = params.texType;
		this->texStr = params.texStr;
		this->texStr1 = 1.f - params.texStr;
		this->clamp = params.clamp;
		this->clampTo = params.clampTo;
	}
};

#endif