/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#pragma once

#include "resource.h"

#include "max.h"
#include <bmmlib.h>
#include "iparamb2.h"
#include "iparamm2.h"
#include "render.h"  
#include "texutil.h"
#include "gizmo.h"
#include "gizmoimp.h"
#include "istdplug.h"

#include "vraygeom.h"
#include "rayserver.h"

#include "primitiveStatic.h"
#include "primitiveMoving.h"

#include "IParticleObjectExt.h"
#include "iFnPub.h"

#if MAX_RELEASE<13900
#include "maxscrpt/MAXScrpt.h"
#else
#include "maxscript/maxscript.h"
const TCHAR *GetString(int id);
#endif

#include "macrorec.h"

//#include "ParticleFlow/IParticleGroup.h"
#define PARTICLEGROUP_INTERFACE Interface_ID(0x2c712d9f, 0x7bc54cb0) // From the include commented out

#define	PLUGIN_CLASSID Class_ID(0x3d1bec1b, 0x43756d05)
#define PLUGIN_INTERFACEID Interface_ID(0x3d1bec1c, 0x43756d06)

#define STR_CLASSNAME _T("BerconMetaball")
#define STR_INTERNALNAME _T("BerconMetaball")
#define STR_LIBDESC _T("BerconMetaball")
#define STR_DLGTITLE _T("BerconMetaball Parameters")

#define REFNO_PBLOCK 0

enum {	
	pb_icon,
	pb_nodes,
	pb_points,
	pb_pflow,
	pb_allPflow,
	pb_size,
	pb_threshold,
	pb_step,
	pb_error,
	pb_fieldFunction,
	pb_useAverage,
	pb_cutoff,
	pb_trim,
	pb_depth,	
	pb_leafLength,
	pb_leafSize,
	pb_texmap,
	pb_mbSamples,
	pb_useScale,
	pb_useTexture,
	pb_texstr,
	pb_textype,
	pb_clamp,
	pb_clampTo,
	pb_pointSizes,
	pb_velocityThreshold,
	pb_color,
	pb_colors,
};

enum {  	
	fn_addnode,
	fn_removenode,
	fn_addpfnode,
	fn_removepfnode,
	fn_pickmode,
	fn_addmode,
	fn_addpfmode,
};

class IBerconMetaball : public FPMixinInterface {
public:
	BEGIN_FUNCTION_MAP
		VFN_1(fn_addnode, fnAddNode, TYPE_INODE);
		VFN_1(fn_removenode, fnRemoveNode, TYPE_INODE);
		VFN_1(fn_addpfnode, fnAddPFNode, TYPE_INODE);
		VFN_1(fn_removepfnode, fnRemovePFNode, TYPE_INODE);
		VFN_0(fn_pickmode, fnPickMode);
		VFN_0(fn_addmode, fnAddMode);
		VFN_0(fn_addpfmode, fnAddPFMode);
	END_FUNCTION_MAP

	FPInterfaceDesc* GetDesc();

	virtual void	fnAddNode(INode *node)=0;
	virtual void	fnRemoveNode(INode *node)=0;
	virtual void	fnAddPFNode(INode *node)=0;
	virtual void	fnRemovePFNode(INode *node)=0;
	virtual void	fnPickMode() = 0;
	virtual void	fnAddMode() = 0;
	virtual void	fnAddPFMode() = 0;
};

class BerconMetaball;

class BerconMetaballValidatorClass : public PBValidator
{
	public:
		BerconMetaball *mod;
	private:
		BOOL Validate(PB2Value &v);
};

class BerconMetaball: public GeomObject, public VR::VRenderObject, public IBerconMetaball {
	Mesh mesh; // An empty dummy mesh returned from GetRenderMesh()	
	INode *selfNode;
	BerconMetaballValidatorClass validator;	

public:
	IParamBlock2 *pblock2;
	IParamMap2 *pmap;
	static IObjParam *ip;
	Interval valid;
	
	BOOL suspendSnap; // Snap suspension flag (TRUE during creation only)	
				
	float simple;
 	int extDispFlags;

	BerconMetaball(void);
	~BerconMetaball(void);

	// Dialog
	HWND hWndParam;
	HWND hWndNode;
	HWND hWndPF;
	BOOL inPickMode;	
	Tab<INode*> pfNodes;
	Tab<int> addPFNodes;

	void PickNodesMode();
	void DisableButtons(HWND hWnd);
	void PickPFEvents(HWND hWnd);
	void RemoveSelectedNode();

	// From IBerconMetaball
	void fnAddNode(INode *node);
	void fnRemoveNode(INode *node);
	void fnAddPFNode(INode *node);
	void fnRemovePFNode(INode *node);
	void fnPickMode();
	void fnAddMode();
	void fnAddPFMode();	

	BaseInterface* GetInterface(Interface_ID id) {
		if (id == PLUGIN_INTERFACEID) return (IBerconMetaball*)this; 
		else return GeomObject::GetInterface(id);
	}
	
	// From BaseObject
	int IsRenderable() { return true; }
	int HitTest(TimeValue t, INode* inode, int type, int crossing, int flags, IPoint2 *p, ViewExp *vpt);
	void Snap(TimeValue t, INode* inode, SnapInfo *snap, IPoint2 *p, ViewExp *vpt);
	void SetExtendedDisplay(int flags);
	int Display(TimeValue t, INode* inode, ViewExp *vpt, int flags);
	CreateMouseCallBack* GetCreateMouseCallBack();
	void BeginEditParams( IObjParam *ip, ULONG flags,Animatable *prev);
	void EndEditParams( IObjParam *ip, ULONG flags,Animatable *next);
	void InvalidateUI();

	// From Object
	ObjectState Eval(TimeValue time);
	void InitNodeName(TSTR& s) { s=STR_CLASSNAME; }
	ObjectHandle ApplyTransform(Matrix3& matrix) { return this; }
	Interval ObjectValidity(TimeValue t);

	// We don't convert to anything
	int CanConvertToType(Class_ID obtype) { return FALSE; }
	Object* ConvertToType(TimeValue t, Class_ID obtype) { assert(0);return NULL; }	
	void GetWorldBoundBox(TimeValue t, INode *mat, ViewExp *vpt, Box3& box);
	void GetLocalBoundBox(TimeValue t, INode *mat, ViewExp *vpt, Box3& box);
	void GetDeformBBox(TimeValue t, Box3 &b, Matrix3 *tm, BOOL useSel);
	int DoOwnSelectHilite()	{ return 1; }

	// From GeomObject
	Mesh* GetRenderMesh(TimeValue t, INode *inode, View& view, BOOL& needDelete);

	// Animatable methods
	void DeleteThis() { delete this; }
	Class_ID ClassID() { return PLUGIN_CLASSID; }
	void GetClassName(TSTR& s) { s=_T("BerconMetaball"); }
	int IsKeyable() { return 0; }
	void* GetInterface(ULONG id);
	void ReleaseInterface(ULONG id, void *ip);
	
	// Direct paramblock access
	int	NumParamBlocks() { return 1; }	
	IParamBlock2* GetParamBlock(int i) { return pblock2; }
	IParamBlock2* GetParamBlockByID(BlockID id) { return (pblock2->ID() == id) ? pblock2 : NULL; }
	int NumSubs() { return 1; }  
	Animatable* SubAnim(int i);
	TSTR SubAnimName(int i);

	// From ref
 	int NumRefs() { return 1; }
	RefTargetHandle GetReference(int i);
	void SetReference(int i, RefTargetHandle rtarg);
	RefTargetHandle Clone(RemapDir& remap=DefaultRemapDir());
	RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message);

	// From VRenderObject
	int init(const ObjectState &os, INode *node, VR::VRayCore *vray);
	VR::VRenderInstance* newRenderInstance(INode *node, VR::VRayCore *vray, int renderID);
	void deleteRenderInstance(VR::VRenderInstance *ri);
	void renderBegin(TimeValue t, VR::VRayCore *vray);
	void renderEnd(VR::VRayCore *vray);
	void frameBegin(TimeValue t, VR::VRayCore *vray);
	void frameEnd(VR::VRayCore *vray);

	// Other methods
	void draw(TimeValue t, INode *node, ViewExp *vpt);
};

class BerconMetaballClassDesc: public ClassDesc2 {
public:
	int IsPublic(void)				{ return TRUE; }
	void *Create(BOOL loading)		{ return new BerconMetaball; }
	const TCHAR *ClassName(void)	{ return STR_CLASSNAME; }
	SClass_ID SuperClassID(void)	{ return GEOMOBJECT_CLASS_ID; }
	Class_ID ClassID(void)			{ return PLUGIN_CLASSID; }
	const TCHAR* Category(void)		{ return _T("VRay"); }
	const TCHAR* InternalName(void) { return STR_INTERNALNAME; }
	HINSTANCE HInstance(void)		{ return hInstance; }
};

class BerconMetaballCreateCallBack: public CreateMouseCallBack {
	BerconMetaball *META;
	IPoint2 sp0;
	Point3 p0;
public:
	int proc(ViewExp *vpt, int msg, int point, int flags, IPoint2 m, Matrix3& mat);
	void SetObj(BerconMetaball *obj) { META=obj; }
};

class BerconMetaballParamsDlgProc : public ParamMap2UserDlgProc {
	BerconMetaball *obj;
public:
	BerconMetaballParamsDlgProc(BerconMetaball *s) { obj = s; }
	INT_PTR DlgProc(TimeValue t,IParamMap2 *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
	void DeleteThis() { delete this; }
};

class BerconMetaballNodesDlgProc : public ParamMap2UserDlgProc {
	BerconMetaball *obj;
public:
	BerconMetaballNodesDlgProc(BerconMetaball *s) { obj = s; }
	INT_PTR DlgProc(TimeValue t,IParamMap2 *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
	void DeleteThis() { delete this; }
};

class BerconMetaballPFlowDlgProc : public ParamMap2UserDlgProc {
	BerconMetaball *obj;
public:
	BerconMetaballPFlowDlgProc(BerconMetaball *s) { obj = s; }
	INT_PTR DlgProc(TimeValue t,IParamMap2 *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
	void DeleteThis() { delete this; }
};

class BerconMetaballTextureDlgProc : public ParamMap2UserDlgProc {
	BerconMetaball *obj;
public:
	BerconMetaballTextureDlgProc(BerconMetaball *s) { obj = s; }
	INT_PTR DlgProc(TimeValue t,IParamMap2 *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
	void DeleteThis() { delete this; }
};

class MyEnumProc : public DependentEnumProc {
	bool mbDoHalt;
public :
	MyEnumProc( bool bDoHalt ) { mbDoHalt = bDoHalt; }
	virtual int proc(ReferenceMaker *rmaker); 
	INodeTab Nodes;              		
};

class PickControlNode : public PickModeCallback, public PickNodeCallback {
public:				
	BerconMetaball *obj;
	HWND hWnd;
	PickControlNode() {obj=NULL;}
	BOOL HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags);		
	BOOL Pick(IObjParam *ip,ViewExp *vpt);		
	void EnterMode(IObjParam *ip);
	void ExitMode(IObjParam *ip);		
	BOOL Filter(INode *node);
	PickNodeCallback *GetFilter() {return this;}
	BOOL RightClick(IObjParam *ip,ViewExp *vpt) {return TRUE;}
	HCURSOR GetDefCursor(IObjParam *ip);
	HCURSOR GetHitCursor(IObjParam *ip);
};

class DumpHitDialog : public HitByNameDlgCallback {
public:
	BerconMetaball *eo;
	DumpHitDialog(BerconMetaball *e) {eo=e;}
	const TCHAR *dialogTitle() {return GetString(IDS_ADD);}
	const TCHAR *buttonText() {return GetString(IDS_ADD);}
	BOOL singleSelect() {return FALSE;}
	BOOL useProc() {return TRUE;}
	int filter(INode *node);
	void proc(INodeTab &nodeTab);
};
