/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#include "maxobject.h"
#include "instanceStatic.h"
#include "instanceMoving.h"
#include "pb2template_generator.h"
#include "defrayserver.h"

IObjParam* BerconMetaball::ip = NULL;
TCHAR *iconText=_T("BerconMetaball");
static BerconMetaballClassDesc classDesc;

/*
	############ Maxscript interface ############
*/

static FPInterfaceDesc vraymetaball_interface(
	PLUGIN_INTERFACEID, _T("BerconMetaballOps"), 0, &classDesc , FP_MIXIN,
	fn_addnode, _T("addMetaball"), 0, TYPE_VOID, 0, 1, _T("node"), 0, TYPE_INODE,
	fn_removenode, _T("removeMetaball"), 0, TYPE_VOID, 0, 1, _T("node"), 0, TYPE_INODE,
	fn_addpfnode, _T("addPFMetaball"), 0, TYPE_VOID, 0, 1, _T("node"), 0, TYPE_INODE,
	fn_removepfnode, _T("removePFMetaball"), 0, TYPE_VOID, 0, 1, _T("node"), 0, TYPE_INODE,
	fn_pickmode, _T("pick"), 0, TYPE_VOID, 0, 0,
	fn_addmode,_T("add"), 0, TYPE_VOID, 0, 0,
	fn_addpfmode, _T("addPF"), 0, TYPE_VOID, 0, 0,
	PB_END
);

FPInterfaceDesc* IBerconMetaball::GetDesc() {
    return &vraymetaball_interface;
}


HINSTANCE hInstance;
int controlsInit=FALSE;

BOOL WINAPI DllMain(HINSTANCE hinstDLL,ULONG fdwReason,LPVOID lpvReserved) {
	hInstance=hinstDLL;

	if (!controlsInit) {
		controlsInit=TRUE;
#if MAX_RELEASE<13900
		InitCustomControls(hInstance);
#endif
		InitCommonControls();
	}

	return(TRUE);
}

__declspec( dllexport ) const TCHAR* LibDescription() { return STR_LIBDESC; }
__declspec( dllexport ) int LibNumberClasses() { return 1; }

__declspec( dllexport ) ClassDesc* LibClassDesc(int i) {
	switch(i) { case 0: return &classDesc; }
	return NULL;
}

__declspec( dllexport ) ULONG LibVersion() { return VERSION_3DSMAX; }

const TCHAR *GetString(int id) {
   static TCHAR buf[256];
   if (hInstance)
      return LoadString(hInstance, id, buf, sizeof(buf)) ? buf : NULL;
   return NULL;
}

enum { params, }; 
enum { rollout_base, rollout_nodes, rollout_pflow, rollout_texture, rollout_about, };

static ParamBlockDesc2 param_blk(params, _T("params"), 0, &classDesc, 
	P_AUTO_CONSTRUCT + P_AUTO_UI + P_MULTIMAP, REFNO_PBLOCK,   
	5,   
	rollout_base,	IDD_PANEL,	IDS_PARAMS, 0, 0, NULL,   
	rollout_texture,IDD_TEXTURE,IDS_TEXPARAM, 0, 1, NULL,
	rollout_nodes,	IDD_NODES,	IDS_NODEPARAMS, 0, 0, NULL,
	rollout_pflow,	IDD_PFLOW,	IDS_PFPARAM, 0, 1, NULL,	
	rollout_about,	IDD_ABOUT,	IDS_ABOUT, 0, 1, NULL,	

	pb_nodes,    _T("nodes"),			TYPE_INODE_TAB,      0, P_AUTO_UI|P_VARIABLE_SIZE, IDS_NODES,
		p_ui,       rollout_nodes, TYPE_NODELISTBOX, IDC_NODES, 0, 0, IDC_REMOVE,
		PB_END,

	pb_points,    _T("metaballs"),			TYPE_POINT3_TAB, 0, P_AUTO_UI|P_VARIABLE_SIZE, 0,
		PB_END,

   pb_allPflow,   _T("allPflow"),		TYPE_BOOL,     0,          IDS_USEALLPFEVENTS,
      p_default,		TRUE,
      p_ui,				rollout_pflow, TYPE_SINGLECHEKBOX,    IDC_USEALLPF, 
      PB_END, 

   pb_pflow,    _T("pflow"),			TYPE_INODE_TAB,      0, P_AUTO_UI|P_VARIABLE_SIZE, IDS_PFEVENTLIST,
      p_ui,				rollout_pflow, TYPE_NODELISTBOX, IDC_PFLIST,0,0,IDC_PFREMOVE,
      PB_END,


	pb_icon,          _T("icon"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_ICON, 
		p_default,     10.f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_ICON_EDIT, IDC_ICON_SPIN, 0.1f, 
		PB_END,

	pb_size,          _T("radius"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_SIZE, 
		p_default,     20.f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_SIZE_EDIT, IDC_SIZE_SPIN, 0.1f, 
		PB_END,
	pb_threshold,          _T("threshold"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_THRES, 
		p_default,     0.5f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_THRES_EDIT, IDC_THRES_SPIN, 0.1f, 
		PB_END,

	pb_step,          _T("step"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_STEP, 
		p_default,     1.f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_STEP_EDIT, IDC_STEP_SPIN, 0.1f, 
		PB_END,
	pb_error,          _T("error"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_ERROR, 
		p_default,     0.0001f, 
		p_range,       0.00001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_ERROR_EDIT, IDC_ERROR_SPIN, 0.1f, 
		PB_END,

	pb_fieldFunction,	_T("fieldFunction"),TYPE_INT,		0, IDS_FIELDFUNCTION,
		p_default,		1,		
		p_ui, 			rollout_base, TYPE_INTLISTBOX, IDC_FIELDFUNCTION, 0, 
		PB_END,

	pb_useAverage,	_T("useAverage"),	TYPE_BOOL,		0, IDS_AVERAGE,
		p_default,		FALSE,
		p_ui,			rollout_base, TYPE_SINGLECHEKBOX, IDC_AVERAGE,
		PB_END,

	pb_cutoff,          _T("cutoff"),	TYPE_FLOAT,    P_ANIMATABLE,  IDS_CUTOFF, 
		p_default,     0.5f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_CUTOFF_EDIT, IDC_CUTOFF_SPIN, 0.1f, 
		PB_END,
	pb_trim,          _T("trim"),		TYPE_FLOAT,    P_ANIMATABLE,  IDS_TRIM, 
		p_default,     0.65f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_TRIM_EDIT, IDC_TRIM_SPIN, 0.1f, 
		PB_END,

	pb_depth,          _T("depth"),       TYPE_INT,    P_ANIMATABLE,  IDS_DEPTH, 
		p_default,     15,
		p_range,       1, 50, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_POS_INT, IDC_DEPTH_EDIT, IDC_DEPTH_SPIN, SPIN_AUTOSCALE, 
		PB_END,
	pb_leafLength,          _T("leafLength"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_LENGTH, 
		p_default,     8.f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_LENGTH_EDIT, IDC_LENGTH_SPIN, 0.1f, 
		PB_END,
	pb_leafSize,          _T("leafSize"),       TYPE_INT,    P_ANIMATABLE,  IDS_LEAF, 
		p_default,     30, 
		p_range,       1, 10000, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_POS_INT, IDC_LEAF_EDIT, IDC_LEAF_SPIN, SPIN_AUTOSCALE, 
		PB_END,

	pb_texmap,		_T("texmap"),		TYPE_TEXMAP,			P_AUTO_UI,	IDS_TEXMAP, // P_OWNERS_REF
		//p_refno,	   REFNO_TEXMAP,
		//p_subtexno,	   0,		
		p_ui,		   rollout_texture,	TYPE_TEXMAPBUTTON, IDC_TEXMAP,
		PB_END,

	pb_mbSamples,          _T("mbSamples"),       TYPE_INT,    P_ANIMATABLE,  IDS_MBSAMPLES, 
		p_default,     2, 
		p_range,       1, 160, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_POS_INT, IDC_MBSAMPLES_EDIT, IDC_MBSAMPLES_SPIN, SPIN_AUTOSCALE, 
		PB_END,

	pb_useScale,	_T("useScale"),	TYPE_BOOL,		0, IDS_USESCALE,
		p_default,		FALSE,
		p_ui,			rollout_base, TYPE_SINGLECHEKBOX, IDC_USESCALE,
		PB_END,

	pb_useTexture,	_T("useTexture"),	TYPE_BOOL,		0, IDS_USETEXTURE,
		p_default,		TRUE,
		p_ui,			rollout_texture, TYPE_SINGLECHEKBOX, IDC_USETEXTURE,
		PB_END,

	pb_textype,	_T("textureType"),TYPE_INT,		0, IDS_TEXTURETYPE,
		p_default,		0,		
		p_ui, 			rollout_texture, TYPE_INTLISTBOX, IDC_FIELDFUNCTION, 0, 
		PB_END,

	pb_clamp,	_T("useClamp"),	TYPE_BOOL,		0, IDS_USECLAMP,
		p_default,		FALSE,
		p_ui,			rollout_texture, TYPE_SINGLECHEKBOX, IDC_CLAMP,
		PB_END,

	pb_texstr,          _T("textureStr"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_TEXTURESTR, 
		p_default,     1.f, 
		p_range,       0.0f, 1000000.0f,
		p_ui,          rollout_texture,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_TEXSTR_EDIT, IDC_TEXSTR_SPIN, 0.1f, 
		PB_END,

	pb_clampTo,          _T("clampTo"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_CLAMPTO, 
		p_default,     1.f, 
		p_range,       0.0001f, 1000000.0f, 
		p_ui,          rollout_texture,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_CLAMP_EDIT, IDC_CLAMP_SPIN, 0.1f, 
		PB_END,

	pb_pointSizes,    _T("metaballSizes"),			TYPE_FLOAT_TAB, 0, P_AUTO_UI|P_VARIABLE_SIZE, 0,
		PB_END,		

	// TODO: This isn't really useful feature if pflow is fixed and there are no more bugs with Velocity channel and Spawn operator
	pb_velocityThreshold,  _T("velocityThreshold"),       TYPE_FLOAT,    P_ANIMATABLE,  IDS_VELOCITYTHRESH, 
		p_default,     -1.f, 
		p_range,       -1.0f, 1000000.0f, 
		p_ui,          rollout_base,	TYPE_SPINNER,	EDITTYPE_FLOAT, IDC_VEL_EDIT, IDC_VEL_SPIN, 0.1f, 
		PB_END,

	pb_color,	_T("color"),TYPE_INT,		0, IDS_FIELDFUNCTION,
		p_default,		0,		
		p_ui, 			rollout_base, TYPE_INTLISTBOX, IDC_FIELDFUNCTION2, 0, 
		PB_END,

	pb_colors,    _T("colors"),			TYPE_POINT3_TAB, 0, P_AUTO_UI|P_VARIABLE_SIZE, 0,
		PB_END,

	PB_END
);

BerconMetaball::BerconMetaball() {
	pblock2 = NULL;
	classDesc.MakeAutoParamBlocks(this);
	suspendSnap = FALSE;
	inPickMode = FALSE;
	selfNode = NULL;
}

BerconMetaball::~BerconMetaball() {
}

void BerconMetaball::InvalidateUI() {
	param_blk.InvalidateUI(); //(pblock2->LastNotifyParamID())
}

void BerconMetaball::BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev) {
	this->ip = ip;

	classDesc.BeginEditParams(ip, this, flags, prev);

	param_blk.SetUserDlgProc(rollout_base,		new BerconMetaballParamsDlgProc(this));	
	param_blk.SetUserDlgProc(rollout_pflow,		new BerconMetaballPFlowDlgProc(this));	
	param_blk.SetUserDlgProc(rollout_nodes,		new BerconMetaballNodesDlgProc(this));
	param_blk.SetUserDlgProc(rollout_texture,	new BerconMetaballTextureDlgProc(this));
	param_blk.ParamOption(pb_nodes,p_validator,&validator);
}

void BerconMetaball::EndEditParams(IObjParam *ip, ULONG flags, Animatable *next) {
   classDesc.EndEditParams(ip, this, flags, next);
   ip->ClearPickMode();
   this->ip = NULL;
}

RefTargetHandle BerconMetaball::Clone(RemapDir& remap) {
	BerconMetaball* newob=new BerconMetaball();	
	BaseClone(this, newob, remap);
	newob->ReplaceReference(0, pblock2->Clone(remap));
	return newob;
}

Animatable* BerconMetaball::SubAnim(int i) {
	switch (i) {
		case 0: return pblock2;
		default: return NULL;
	}
}

TSTR BerconMetaball::SubAnimName(int i) {
	switch (i) {
		case 0: return STR_DLGTITLE;
		default: return _T("");
	}
}

RefTargetHandle BerconMetaball::GetReference(int i) {
	switch (i) {
		case REFNO_PBLOCK: return pblock2;
		default: return NULL;
	}
}

void BerconMetaball::SetReference(int i, RefTargetHandle rtarg) {
	switch (i) {
		case REFNO_PBLOCK: pblock2 = (IParamBlock2*)rtarg; break;
	}
}

RefResult BerconMetaball::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message) {
	switch (message) {
		case REFMSG_CHANGE:
			if (hTarget==pblock2) {
				param_blk.InvalidateUI();
			}
			break;
	}
	return REF_SUCCEED;
}

Interval BerconMetaball::ObjectValidity(TimeValue t) {
	Interval valid=FOREVER;
	float f;
	pblock2->GetValue(pb_icon, t, f, valid);
	return valid;
}

int BerconMetaballCreateCallBack::proc(ViewExp *vpt, int msg, int point, int flags, IPoint2 m, Matrix3& mat) {
	if (!META) return CREATE_ABORT;

	Point3 np=vpt->SnapPoint(m,m,NULL,SNAP_IN_PLANE);

	switch (msg) {
		case MOUSE_POINT:
			switch (point) {
				case 0:
					META->suspendSnap=TRUE;				
					sp0=m;
					p0=vpt->SnapPoint(m, m, NULL, SNAP_IN_3D);
					mat.SetTrans(p0);
					META->pblock2->SetValue(pb_icon, 0, 0.0f);
					//return CREATE_STOP;
					return CREATE_CONTINUE;
				case 1:
					float r=Length(np-mat.GetTrans());
					if (r<1e-3f) return CREATE_ABORT;
					META->pblock2->SetValue(pb_icon, 0, r);
					return CREATE_STOP;
			}
			return CREATE_CONTINUE;

		case MOUSE_MOVE:
			if (point==1) {
				float r=Length(np-mat.GetTrans());
				META->pblock2->SetValue(pb_icon, 0, r);
			}
			return CREATE_CONTINUE;

		case MOUSE_ABORT:
			return CREATE_ABORT;
	}

	return CREATE_CONTINUE;
}

static BerconMetaballCreateCallBack createCB;

CreateMouseCallBack* BerconMetaball::GetCreateMouseCallBack() {
	createCB.SetObj(this);
	return &createCB;
}

void BerconMetaball::SetExtendedDisplay(int flags) {
	extDispFlags = flags;
}

void BerconMetaball::GetLocalBoundBox(TimeValue t, INode* inode, ViewExp* vpt, Box3& box) {
	box.Init();
	float radius = pblock2->GetFloat(pb_icon, t);
	box+=Point3(-radius, -radius, -radius);
	box+=Point3(radius, radius, radius);
}

void BerconMetaball::GetWorldBoundBox(TimeValue t, INode* inode, ViewExp* vpt, Box3& box) {
	if (!inode) return;
	Box3 localBox;
	GetLocalBoundBox(t, inode, vpt, localBox);
	box=localBox*(inode->GetObjectTM(t));
}

void BerconMetaball::GetDeformBBox(TimeValue t, Box3 &b, Matrix3 *tm, BOOL useSel) {
	if (!tm) {
		GetLocalBoundBox(t, NULL, NULL, b);
	} else {
		Box3 bbox;
		GetLocalBoundBox(t, NULL, NULL, bbox);
		b.Init();
		for (int i=0; i<8; i++) b+=(*tm)*bbox[i];
	}
}

int BerconMetaball::HitTest(TimeValue t, INode *node, int type, int crossing, int flags, IPoint2 *p, ViewExp *vpt) {
	static HitRegion hitRegion;
	DWORD	savedLimits;

	GraphicsWindow *gw=vpt->getGW();	
	Material *mtl=gw->getMaterial();
	MakeHitRegion(hitRegion, type, crossing, 4, p);

	gw->setRndLimits(((savedLimits = gw->getRndLimits())|GW_PICK)&~GW_ILLUM);
	gw->setHitRegion(&hitRegion);
	gw->clearHitCode();

	draw(t, node, vpt);

	gw->setRndLimits(savedLimits);
	
	if((hitRegion.type != POINT_RGN) && !hitRegion.crossing) return TRUE;
	return gw->checkHitCode();
}

void BerconMetaball::Snap(TimeValue t, INode* inode, SnapInfo *snap, IPoint2 *p, ViewExp *vpt) {
	if (suspendSnap) return;
}

inline void drawLine(GraphicsWindow *gw, Point3 &p0, Point3 &p1) {
	Point3 p[3]={ p0, p1 };
	gw->segment(p, TRUE);
}

inline void drawTriangle(GraphicsWindow *gw, Point3 &p0, Point3 &p1, Point3 &p2) {
	Point3 n=Normalize((p1-p0)^(p2-p0));
	Point3 p[4]={ p0, p1, p2 };
	Point3 uvw[4]={ Point3(0,0,0), Point3(0,0,0), Point3(0,0,0) };
	Point3 nrm[4]={ n, n, n };
	gw->triangleN(p, nrm, uvw);
}

inline void drawBBox(GraphicsWindow *gw, Box3 &b) {
	gw->setTransform(Matrix3(1));
	Point3 p[8];
	for (int i=0; i<8; i++) p[i]=b[i];
	gw->startSegments();

	drawLine(gw, p[0], p[1]);
	drawLine(gw, p[0], p[2]);
	drawLine(gw, p[0], p[4]);

	/*
	drawLine(gw, p[7], p[6]);
	drawLine(gw, p[7], p[5]);
	drawLine(gw, p[7], p[3]);
	*/
	gw->endSegments();
}

//const float size=40.0f;

void BerconMetaball::draw(TimeValue t, INode *node, ViewExp *vpt) {
	GraphicsWindow *gw=vpt->getGW();

	Matrix3 tm=node->GetObjectTM(t);
	/*
	tm.NoScale();
	float scaleFactor=vpt->NonScalingObjectSize()*vpt->GetVPWorldWidth(tm.GetTrans())/(float)360.0;
	tm.Scale(Point3(scaleFactor,scaleFactor,scaleFactor));
	*/
	gw->setTransform(tm);

	Color color=Color(node->GetWireColor());
	if (node->IsFrozen()) color=GetUIColor(COLOR_FREEZE);
	else if (node->Selected()) color=GetUIColor(COLOR_SELECTION);
	gw->setColor(LINE_COLOR, color);

	/*
	Point3 p[6];
	p[0]=Point3(-iconSize, -iconSize, 0.0f);
	p[1]=Point3(iconSize, -iconSize, 0.0f);
	p[2]=Point3(iconSize, iconSize, 0.0f);
	p[3]=Point3(-iconSize, iconSize, 0.0f);
	p[4]=Point3(0.0f, 0.0f, 0.0f);
	p[5]=Point3(0.0f, 0.0f, iconSize);

	gw->setColor(FILL_COLOR, color);
	gw->startTriangles();

	drawTriangle(gw, p[0], p[1], p[2]);
	drawTriangle(gw, p[2], p[3], p[0]);

	gw->endTriangles();

	gw->setColor(LINE_COLOR, color);
	gw->startSegments();

	drawLine(gw, p[0], p[1]);
	drawLine(gw, p[1], p[2]);
	drawLine(gw, p[2], p[3]);
	drawLine(gw, p[3], p[0]);
	drawLine(gw, p[0], p[2]);
	drawLine(gw, p[1], p[3]);
	drawLine(gw, p[4], p[5]);
	drawLine(gw, p[5], p[5]*0.8f+p[0]*0.2f);
	drawLine(gw, p[5], p[5]*0.8f+p[1]*0.2f);
	drawLine(gw, p[5], p[5]*0.8f+p[2]*0.2f);
	drawLine(gw, p[5], p[5]*0.8f+p[3]*0.2f);

	gw->endSegments();
	*/

	float radius = pblock2->GetFloat(pb_icon, t);
	int nsegs=30;
	float u0=radius, v0=0.0f;
	Point3 pt[3];

	gw->startSegments();

	for (int i=0; i<nsegs; i++) {
		float a=2.0f*(float)pi*float(i+1)/float(nsegs);

		float u1=radius*cosf(a);
		float v1=radius*sinf(a);

		pt[0]=Point3(u0, v0, 0.0f);
		pt[1]=Point3(u1, v1, 0.0f);
		gw->segment(pt, true);

		pt[0]=Point3(0.0f, u0, v0);
		pt[1]=Point3(0.0f, u1, v1);
		gw->segment(pt, true);

		pt[0]=Point3(u0, 0.0f, v0);
		pt[1]=Point3(u1, 0.0f, v1);
		gw->segment(pt, true);

		u0=u1;
		v0=v1;
	}

	gw->endSegments();

	/*
	if (hitTest && gw->checkHitCode()) {
		*res=TRUE;
		vpt->CtrlLogHit(inode, gw->getHitDistance(), 0, 0);
		if (abortOnHit) break;
		gw->clearHitCode();
	}
	*/

	tm.NoScale();
	float scaleFactor=vpt->NonScalingObjectSize()*vpt->GetVPWorldWidth(tm.GetTrans())/(float)360.0;
	tm.Scale(Point3(scaleFactor,scaleFactor,scaleFactor));
	gw->setTransform(tm);

	IPoint3 ipt;
	Point3 org(0,0,0);
	gw->wTransPoint(&org, &ipt);

	SIZE sp;
	gw->getTextExtents(iconText, &sp);

	ipt.x-=sp.cx/2;
	ipt.y-=sp.cy/2;

	gw->setColor(TEXT_COLOR, 0.0f, 0.0f, 0.0f);
	gw->wText(&ipt, iconText);

	ipt.x--;
	ipt.y--;
	gw->setColor(TEXT_COLOR, 1.0f, 1.0f, 1.0f);
	gw->wText(&ipt, iconText);
}

int BerconMetaball::Display(TimeValue t, INode* node, ViewExp *vpt, int flags) {
	draw(t, node, vpt);
	return 0;
}

ObjectState BerconMetaball::Eval(TimeValue time) {	
	return ObjectState(this);
}

void* BerconMetaball::GetInterface(ULONG id) {
	if (id==I_VRAYGEOMETRY) return (VR::VRenderObject*) this;
	return GeomObject::GetInterface(id);
}

void BerconMetaball::ReleaseInterface(ULONG id, void *ip) {
	if (id==I_VRAYGEOMETRY) return;
	GeomObject::ReleaseInterface(id, ip);
}

Mesh* BerconMetaball::GetRenderMesh(TimeValue t, INode *inode, View& view, BOOL& needDelete) {	
	needDelete=false;
	return &mesh;
}

// VRenderObject stuff here

int BerconMetaball::init(const ObjectState &os, INode *node, VR::VRayCore *vray) {
	VRenderObject::init(os, node, vray);
	return true;
}

void BerconMetaball::renderBegin(TimeValue t, VR::VRayCore *vray) {
	VRenderObject::renderBegin(t, vray);
}

void BerconMetaball::renderEnd(VR::VRayCore *vray) {
	VRenderObject::renderEnd(vray);
}

void BerconMetaball::frameBegin(TimeValue t, VR::VRayCore *vray) {
	VRenderObject::frameBegin(t, vray);
}

void BerconMetaball::frameEnd(VR::VRayCore *vray) {
	VRenderObject::frameEnd(vray);
}

VR::VRenderInstance* BerconMetaball::newRenderInstance(INode *node, VR::VRayCore *vray, int renderID) {
	
	bool motionBlur = vray->getSequenceData().params.moblur.on && (pblock2->GetInt(pb_mbSamples, 0) - 1); // <- TODO: Time zero here, could be something else
	if (motionBlur) {
		return new BerconMetaballInstanceMoving(this, node, vray, renderID);
	} else {
		return new BerconMetaballInstanceStatic(this, node, vray, renderID);
	}	
}

void BerconMetaball::deleteRenderInstance(VR::VRenderInstance *ri) {
	if (((BerconMetaballInstance *) ri)->dynamicType)
		delete (BerconMetaballInstanceMoving*) ri;
	else
		delete (BerconMetaballInstanceStatic*) ri;
}
