/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/
#include "max.h"
#include "BVHMoving.h"

using namespace VR;

static Box calcBox(StaticBox b[2], real time) {
	Box ib;
	ib.pmin=b[0].pmin+b[1].pmin*time;
	ib.pmax=b[0].pmax+b[1].pmax*time;
	return ib;
}

static Box getBoundingBox(StaticBox b[2]) {
	Box b0(b[0]);
	Box b1(b[0].pmin+b[1].pmin, b[0].pmax+b[1].pmax);
	return b0+b1;
}

/*
	############ BVHMContainer ############
*/

void BVHMContainer::calculatePositions(Vector* pointCloud[2], real time) {	
	noPositions = false;
	count = leaf->count;
	cloud = new Vector[count];
	for (int i=0;i<count;i++)
		cloud[i] = pointCloud[0][leaf->points[i]] + time * pointCloud[1][leaf->points[i]];
}

void BVHMContainer::calculatePositions(Vector* pointCloud[2], VR::Ireal* radii[2], real time, bool linear) {	
	noPositions = false;
	count = leaf->count;
	cloud = new Vector[count];
	sizes = new Ireal[count];
	if (linear) {
		sizesLin = new Ireal[count];
		for (int i=0;i<count;i++) {
			cloud[i] = pointCloud[0][leaf->points[i]] + time * pointCloud[1][leaf->points[i]];
			sizesLin[i] = radii[0][leaf->points[i]] + time * radii[1][leaf->points[i]];
			sizes[i] = sizesLin[i] * sizesLin[i];
		}
	} else {
		for (int i=0;i<count;i++) {
			cloud[i] = pointCloud[0][leaf->points[i]] + time * pointCloud[1][leaf->points[i]];
			sizes[i] = radii[0][leaf->points[i]] + time * radii[1][leaf->points[i]];
			sizes[i] *= sizes[i];
		}
	}
}

void BVHMContainer::getField(Vector* pointCloud[2], real time, FieldMoving* field) {
	if (noPositions) calculatePositions(pointCloud, time);
	field->addPoints(cloud, count);
}

void BVHMContainer::getFieldGrad(Vector* pointCloud[2], real time, FieldGradMoving* grad) {
	if (noPositions) calculatePositions(pointCloud, time); // Probably not required, once we get to grad computation, positions are already computed when searching intersection
	grad->addPoints(cloud, leaf->points, count);	
}

void BVHMContainer::getField(Vector* pointCloud[2], VR::Ireal* radii[2], real time, FieldMoving* field, bool linear) {
	if (noPositions) calculatePositions(pointCloud, radii, time, linear);
	field->addPoints(cloud, sizes, sizesLin, count);
}

void BVHMContainer::getFieldGrad(Vector* pointCloud[2], VR::Ireal* radii[2], real time, FieldGradMoving* grad, bool linear) {
	if (noPositions) calculatePositions(pointCloud, radii, time, linear); // Probably not required, once we get to grad computation, positions are already computed when searching intersection
	grad->addPoints(cloud, sizes, sizesLin, leaf->points, count);	
}

/*
	############ BVHMRay ############
*/

// TODO Optimize search, now brute force
void BVHMRay::add(Ireal min, Ireal max, BVHMContainer* cont) {
	steps.push_back(new BVHMStep(min, max, cont, NULL));
	
	/*
	if (this->first == NULL) {
		this->first = new BVHMStep(min, max, cont, NULL);
		return;
	}

	if (min < this->first->min) {
		this->first = new BVHMStep(min, max, cont, this->first);
		return;
	}

	BVHMStep* cur = this->first;
	while (cur->next != NULL && min > cur->next->min)
		cur = cur->next;
	cur->next = new BVHMStep(min, max, cont, cur->next);
	*/
}

void BVHMRay::sortSteps() {
	std::sort(steps.begin(), steps.end(), BVHMStepSort());	
}

void BVHMRay::collapseSteps() {
	if (steps.size() == 0) return;
	
	this->first = steps[0];

	for (int i=1; i<steps.size(); i++)
		steps[i-1]->next = steps[i];

	steps.clear();
}

/*
	############ BVHMNode ############
*/

int BVHMNode::intersect(IRay* ray, real time) {
	if (ray->testIntersectBox(calcBox(b, time))) {
		if (!this->type())
			return true;
		if (this->getLeft()->intersect(ray, time))
			return true;
		if (this->getRight()->intersect(ray, time))
			return true;
	}	
	return false;
}

void BVHMNode::initBoundingBox(Vector* pointCloud[2], Ireal* radii[2], int* points, int count, real size) {
	// Create bounding box
	b[0].init();
	b[1].init()	;
	if (radii[0]) {
		for (int i=0;i<count;i++) {
			b[0].addExpanded(pointCloud[0][points[i]], radii[0][points[i]]);			
			b[1].addExpanded(pointCloud[0][points[i]]+pointCloud[1][points[i]], radii[0][points[i]]+radii[1][points[i]]);
		}
	} else {
		for (int i=0;i<count;i++) {						
			b[0].addExpanded(pointCloud[0][points[i]], size);
			b[1].addExpanded(pointCloud[0][points[i]]+pointCloud[1][points[i]], size);
		}
	}
	b[1].pmin -= b[0].pmin;
	b[1].pmax -= b[0].pmax;
}

/*
	############ BVHMBranch ############
*/

void BVHMBranch::traceRay(IRay* ray, BVHMRay* bRay, real time) {
	if (ray->testIntersectBox(calcBox(b, time))) {
		this->getLeft()->traceRay(ray, bRay, time);
		this->getRight()->traceRay(ray, bRay, time);
	}
}

BVHMBranch::BVHMBranch(Vector* pointCloud[2], Ireal* radii[2], int* points, int count, real size, int maxDepth, int depth, int leafSize, real maxLength) {	
	// Create bounding box
	initBoundingBox(pointCloud, radii, points, count, size);

	// Choose which axis to split and where
	Box bb = getBoundingBox(b);
	int axis = bb.maxDimension();
	real middle = (bb.pmin[axis] + bb.pmax[axis]) * 0.5f;
	
	// Count number of points on both sides
	int leftCount = 0;
	for (int i=0;i<count;i++)
		if (pointCloud[0][points[i]][axis] + 0.5 * pointCloud[1][points[i]][axis] < middle)
			leftCount++;					
	int rightCount = count - leftCount;

	// If the other side gets 0 points, all the points are at the same location
	// and there is no longer point dividing them any futher so we just make
	// leaves here
	
	bool forceBVHMLeaf = false;
	if (leftCount == 0 || rightCount == 0) {
		leftCount = count / 2;
		rightCount = count - leftCount;
		forceBVHMLeaf = true;
	}

	// Create fitting arrays for both
	int* leftPoints;
	int* rightPoints;	
	//if (leftCount > 0)
		leftPoints = new int[leftCount];
	//if (rightCount > 0)
		rightPoints = new int[rightCount];
	
	// Assign points to for both sides	
	if (forceBVHMLeaf) {
		int j = 0;
		for (int i=0;i<count;i++)
			if (i < leftCount)
				leftPoints[i] = points[i];			
			else
				rightPoints[j++] = points[i];
	} else {
		int j = 0, k = 0;
		for (int i=0;i<count;i++)
			if (pointCloud[0][points[i]][axis] + 0.5 * pointCloud[1][points[i]][axis] < middle) {
				//if (j < leftCount)
					leftPoints[j++] = points[i];
			} else {
				//if (k < rightCount)
					rightPoints[k++] = points[i];
			}
	}

	// Free previous point array, as its now split into left and right
	delete[] points;

	// Leaves or not?
	real length = bb.pmax[axis] - bb.pmin[axis];

	if (maxDepth <= depth || ((length < maxLength || radii[0]) && leftCount < leafSize) || forceBVHMLeaf) // Make leaf
		this->left = new BVHMLeaf(pointCloud, radii, leftPoints, leftCount, size);
	else // BVHMBranch more		
		this->left = new BVHMBranch(pointCloud, radii, leftPoints, leftCount, size, maxDepth, depth+1, leafSize, maxLength);

	if (maxDepth <= depth || ((length < maxLength || radii[0]) && rightCount < leafSize) || forceBVHMLeaf) // Make leaf
		this->right = new BVHMLeaf(pointCloud, radii, rightPoints, rightCount, size);
	else // BVHMBranch more		
		this->right = new BVHMBranch(pointCloud, radii, rightPoints, rightCount, size, maxDepth, depth+1, leafSize, maxLength);
}

/*
	############ BVHMLeaf ############
*/

void BVHMLeaf::traceRay(IRay* ray, BVHMRay* bRay, real time) {
	IRay ray2 = IRay(*ray);
	if (ray2.intersectBox(calcBox(b, time)))
		bRay->add(ray2.cmint, ray2.cmaxt, new BVHMContainer(this));			
}

BVHMLeaf::BVHMLeaf(Vector* pointCloud[2], Ireal* radii[2], int* points, int count, real size) {
	// Create bounding box
	initBoundingBox(pointCloud, radii, points, count, size);

	this->points = points;
	this->count = count;		
}

/*
	############ BoundingVolumeHierarchy ############
*/

void BoundingVolumeHierarchyMoving::create(Vector* pointCloud[2], Ireal* radii[2], int count, real size, int maxDepth, int leafSize, real maxLength) {
	delete this->root;
	int* points = new int[count];
	for (int i=0;i<count;i++)
		points[i] = i;
	this->root = new BVHMBranch(pointCloud, radii, points, count, size, maxDepth, 1, leafSize, maxLength);
}